**************************************************************
* Read in data
**************************************************************

set more off
clear
* Put name of your main directory path here:
local main = "___"
cd `main'
use data_quarterly.dta, clear

***************************************************************
* Set color scheme (optional)
***************************************************************

set scheme s1color

**************************************************************
* Data work
**************************************************************

* Sample

forval s = 1/4 {
	g sample`s' = 0
}
replace sample1 = 1 if (t>=tq(1960q1)) & (t<=tq(1972q4))
replace sample2 = 1 if (t>=tq(1973q1)) & (t<=tq(1984q4))
replace sample3 = 1 if (t>=tq(1985q1)) & (t<=tq(2019q4))
replace sample4 = 1 if (t>=tq(2020q1)) & (t<=tq(2022q3))

* Overall sample
keep if (t>=tq(1968q1)) & (t<=tq(2022q3))

* Computation of trend productivity growth variable
*** Browse the productivity variable
des pty
tsfilter hp pty_cyc1 = pty, smooth(1600) trend(pty_tr1)
tsfilter hp pty_cyc2 = pty, smooth(16000) trend(pty_tr2)
label var pty "Gth Q/Q SAAR of Labor Pty Nonfarm Business Sector"
label var pty_tr1  "trend component from hp filter (1600)"
label var pty_tr2  "trend component from hp filter (16000)"
* Pty growth
g p			= pty_tr2
* 4-quarter average
gen p_4qma	= (1/4)*(p + p[_n-1] + p[_n-2] + p[_n-3])
label variable p_4qma "Trend pty growth(lambda=16000), 4-quarter avg."

* Drop outlier based on scatter plot
tab gw if t==tq(1972q1)
drop if t==tq(1972q1)

**************************************************************
//* Estimation: Price Inflation
**************************************************************

* Inflation gap
cap drop y x* z*
g y		= Median-pe

* Relative price fluctuations (supply fluctuations)
g z			= CPI - Median
* 4-quater average
gen z_4qa	= (1/4)*(z + z[_n-1] + z[_n-2] + z[_n-3])

* Measure of slack
foreach var in vu {
	* Define measure of slack ("x")
	cap drop x*
	g x		= `var'_4qa
	g x2	= x^2
	g x3	= x^3
	* Compute nonlinear "z" terms
	g z_4qa2 = z_4qa^2
	g z_4qa3 = z_4qa^3	
}

* Estimate equation: either for samples 3 and 4 or just sample 3 (comment out the other with "*")
* Price PC:
newey y x x2 x3 z_4qa z_4qa2 z_4qa3 if (sample3==1|sample4==1), lag(4)

* Print variance-covariance matrix
estat vce

**************************************************************
* Composite contribution (or slope) of PC vis-a-vis V/U + constant
**************************************************************
//*
preserve
local index = 1
g beta	= .
g hi	= .
g lo	= .
sum x if e(sample)

local min	= r(min)
local max	= r(max)
g xs		= .
local n		= 1
forval x = `min'(0.025)`max' {
	local x2 = (`x')^2
	local x3 = (`x')^3
	* Conribution:
	lincom _b[_cons] + _b[x]*`x' + (_b[x2]*`x2') + (_b[x3]*`x3')
	local beta 		= r(estimate)
	local se		= r(se)
	replace beta 	= `beta'			if _n==`n'
	replace hi		= `beta'+1.96*`se'	if _n==`n'	
	replace lo		= `beta'-1.96*`se'	if _n==`n'		
	replace xs	 	= `x'				if _n==`n'
	local n			= `n'+1
}
clist xs beta hi lo if xs~=., noobs
twoway (line beta hi lo xs, yline(0, lcolor(ebg)) lcolor(midblue midblue midblue) lpattern(solid dash dash) lwidth(medthick medthick medthick) ///
legend(off) title("A. P Inflation Gap vs. V/U") xtitle("V/U (4-quarter average)") saving(`index'.gph, replace))

* Save variables for use later on
keep xs beta hi lo 
keep if xs~=. 
foreach var in beta hi lo {
	rename `var' P_VU_`var'
}
sort xs
save P_VU.dta, replace
restore
*/

**************************************************************
* Contribution or Composite slope of PC vis-a-vis "H"
**************************************************************

preserve
local index = 2
g beta	= .
g hi	= .
g lo	= .
sum z_4qa if e(sample)
local min	= r(min)
local max	= r(max)
g xs		= .
local n		= 1
forval x = -3.45(0.05)3.70 {	
	local x2 = (`x')^2
	local x3 = (`x')^3
	* Conribution:
    lincom _b[z_4qa]*`x' + (_b[z_4qa2]*`x2') + (_b[z_4qa3]*`x3')
	local beta 		= r(estimate)
	local se		= r(se)
	replace beta 	= `beta'			if _n==`n'
	replace hi		= `beta'+1.96*`se'	if _n==`n'	
	replace lo		= `beta'-1.96*`se'	if _n==`n'		
	replace xs	 	= `x'				if _n==`n'
	local n			= `n'+1
}

clist xs beta hi lo if xs~=., noobs
twoway (line beta hi lo xs, xline(0, lcolor(ebg)) yline(0, lcolor(ebg)) lcolor(midblue midblue midblue) lpattern(solid dash dash) lwidth(medthick medthick medthick) ///
legend(off) title("C. P Inflation Gap vs. HIS") xtitle("H (4-quarter average)") saving(`index'.gph, replace))
* Save variables for use later on
keep xs beta hi lo 
keep if xs~=. 
foreach var in beta hi lo {
	rename `var' P_H_`var'
}
sort xs
save P_H.dta, replace
restore
*/

**************************************************************
* Estimation: Wage Inflation
**************************************************************

* Inflation gap
cap drop y x* z*
g y		= gw-pe

* Relative price fluctuations (supply fluctuations)
g z			= CPI - Median
* 4-quater average
gen z_4qa	= (1/4)*(z + z[_n-1] + z[_n-2] + z[_n-3])

* Measure of slack
foreach var in vu {
	* Define measure of slack ("x")
	cap drop x*
	g x		= `var'_4qa
	g x2	= x^2
	g x3	= x^3
	* Compute nonlinear "z" terms
	g z_4qa2 = z_4qa^2
	g z_4qa3 = z_4qa^3	
}

* Estimate equation: 
* Wage PC:
newey y x x2 x3 z_4qa z_4qa2 z_4qa3 p_4qma if (sample3==1|sample4==1), lag(4)

* Print variance-covariance matrix
estat vce

**************************************************************
* Composite contribution (or slope) of PC vis-a-vis V/U + constant 
* Also need to add productivity term at sample mean
**************************************************************
//*
preserve
local index = 3
g beta	= .
g hi	= .
g lo	= .
sum x if e(sample)
local min	= r(min)
local max	= r(max)
sum p_4qma if e(sample)
local p_4qma = r(mean)
g xs		= .
local n		= 1
forval x = `min'(0.025)`max' {
	local x2 = (`x')^2
	local x3 = (`x')^3
	* Conribution:
	lincom _b[_cons] + _b[x]*`x' + (_b[x2]*`x2') + (_b[x3]*`x3') + _b[p_4qma]*`p_4qma'
	local beta 		= r(estimate)
	local se		= r(se)
	replace beta 	= `beta'			if _n==`n'
	replace hi		= `beta'+1.96*`se'	if _n==`n'	
	replace lo		= `beta'-1.96*`se'	if _n==`n'		
	replace xs	 	= `x'				if _n==`n'
	local n			= `n'+1
}
clist xs beta hi lo if xs~=., noobs
* Merge in price PC results
sort xs
merge xs using P_VU.dta
drop _merge
label variable beta "Wage inflation gap"
label variable P_VU_beta "Price inflation gap"
twoway ///
(rarea P_VU_hi P_VU_lo xs, color(red%25)) ///
(rarea hi lo xs, color(blue%25)) ///
(line P_VU_beta beta xs, yline(0, lcolor(ebg)) lcolor(red blue) lpattern(dash solid) lwidth(medthick medthick medthick) ///
legend(row(1) symxsize(4) ring(0) position(12) bmargin(none) region(style(none))) ///
legend(order(4 3)) ///
legend(size(medium)) ///
title("A. Inflation Gap vs. V/U") xtitle("V/U (4-quarter average)") saving(`index'.gph, replace))
rename P_VU_hi price_hi
rename P_VU_lo price_lo
rename hi wage_hi
rename lo wage_lo
rename xs VU
rename beta wage_midpoint
rename P_VU_beta price_midpoint
keep if VU~=.
outsheet price_midpoint price_hi price_lo wage_midpoint wage_hi wage_lo VU using "results\charts\Figure 10 Panel A.xls", replace
restore
*/

**************************************************************
* Contribution or Composite slope of PC vis-a-vis "H"
**************************************************************

preserve
local index = 4
g beta	= .
g hi	= .
g lo	= .
sum z_4qa if e(sample)
local min	= r(min)
local max	= r(max)
g xs		= .
local n		= 1
* Pick range to exactly match monthly chart (for ease of comparability)
forval x = -3.45(0.05)3.70 {	
	local x2 = (`x')^2
	local x3 = (`x')^3
	* Conribution:
    lincom _b[z_4qa]*`x' + (_b[z_4qa2]*`x2') + (_b[z_4qa3]*`x3')
	local beta 		= r(estimate)
	local se		= r(se)
	replace beta 	= `beta'			if _n==`n'
	replace hi		= `beta'+1.96*`se'	if _n==`n'	
	replace lo		= `beta'-1.96*`se'	if _n==`n'		
	replace xs	 	= `x'				if _n==`n'
	local n			= `n'+1
}
* Merge in price PC results
sort xs
merge xs using P_H.dta
drop _merge
label variable beta "Wage inflation gap"
label variable P_H_beta "Price inflation gap"
twoway ///
(rarea P_H_hi P_H_lo xs, color(red%25)) ///
(rarea hi lo xs, color(blue%25)) ///
(line P_H_beta beta xs, yline(0, lcolor(ebg)) lcolor(red blue) lpattern(dash solid) lwidth(medthick medthick medthick) ///
legend(row(1) symxsize(4) ring(0) position(12) bmargin(none) region(style(none))) ///
legend(order(4 3)) ///
legend(size(medium)) ///
title("B. Inflation Gap vs. Headline-inflation Shock") xtitle("H (4-quarter average)") saving(`index'.gph, replace))
clist xs beta hi lo if xs~=., noobs
rename P_H_hi price_hi
rename P_H_lo price_lo
rename hi wage_hi
rename lo wage_lo
rename xs H
rename beta wage_midpoint
rename P_H_beta price_midpoint
keep if H~=.
outsheet price_midpoint price_hi price_lo wage_midpoint wage_hi wage_lo H using "results\charts\Figure 10 Panel B.xls", replace
restore
*/

graph combine 3.gph 4.gph, row(2) ysize(8)
